<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') exit;

include_once __DIR__ . '/config.php';
include_once __DIR__ . '/connection.php';
include_once __DIR__ . '/mt5_api/mt5_api.php'; // MT5 API

$input = json_decode(file_get_contents("php://input"), true);
$emailInput = trim($input['email'] ?? '');
$tierLevel = isset($input['level']) ? intval($input['level']) : 1;

if (empty($emailInput)) {
    echo json_encode(["status" => "error", "message" => "Email is required"]);
    exit;
}
if ($tierLevel < 1 || $tierLevel > 20) {
    echo json_encode(["status" => "error", "message" => "Level must be between 1 and 20"]);
    exit;
}

$ibColumn = "ib" . $tierLevel;

$totals = [
    "total_balance" => 0,
    "total_deposit" => 0,
    "total_withdraw" => 0,
    "total_volume" => 0,
    "total_rebate" => 0,
    "total_clients" => 0,
    "total_profit_commission" => 0
];

// --- Fetch all clients under this IB tier ---
$stmt_acc = $conn->prepare("SELECT * FROM liveaccounts WHERE $ibColumn = ?");
$stmt_acc->bind_param("s", $emailInput);
$stmt_acc->execute();
$res_acc = $stmt_acc->get_result();

$tradeIds = [];
$clientsData = [];
$uniqueEmails = [];

while ($acc = $res_acc->fetch_assoc()) {
    $trade_id = intval($acc['trade_id']);
    $tradeIds[] = $trade_id;
    $clientsData[$trade_id] = $acc;
    $uniqueEmails[$acc['email']] = true; // track unique emails
}
$stmt_acc->close();

if (empty($tradeIds)) {
    echo json_encode([
        "status" => "error",
        "message" => "No clients found for this email in this level.",
        "data" => [],
        "totals" => $totals
    ]);
    exit;
}

// --- Connect MT5 ---
$mt5 = new MTWebAPI("WebAPITesterLocal", "./logs", true);
$ret = $mt5->Connect(MT5_SERVER_IP, MT5_SERVER_PORT, 5, MT5_SERVER_WEB_LOGIN, MT5_SERVER_WEB_PASSWORD);
if ($ret != MTRetCode::MT_RET_OK) {
    echo json_encode(["status" => "error", "message" => "MT5 connection failed"]);
    exit;
}

// --- Determine Rank based on number of unique emails (clients) ---
$totalActiveClients = count($uniqueEmails);
$rankNumber = 1;
$rankName = "Rank 1";

$stmt_rank = $conn->prepare("SELECT * FROM ranks WHERE min_active_clients <= ? ORDER BY rank_number DESC LIMIT 1");
$stmt_rank->bind_param("i", $totalActiveClients);
$stmt_rank->execute();
$res_rank = $stmt_rank->get_result();
$rankData = [];
if ($row_rank = $res_rank->fetch_assoc()) {
    $rankNumber = intval($row_rank['rank_number']);
    $rankName = $row_rank['rank_name'];
    $rankData = $row_rank;
}
$stmt_rank->close();

// --- Process each client in current level ---
$clients = [];
foreach ($tradeIds as $trade_id) {
    $acc = $clientsData[$trade_id];
    $email = $acc['email'];

    $balance = 0;
    $account_info = null;
    if ($mt5->UserAccountGet($trade_id, $account_info) == MTRetCode::MT_RET_OK) {
        $balance = floatval($account_info->Balance ?? 0);
    }

    // --- Fetch volume from MT5 ---
    $volume = 0;
    $totalDeals = 0;
    $deals = [];
    $ret_total = $mt5->DealGetTotal($trade_id, 0, 0, $totalDeals);
    if ($ret_total == MTRetCode::MT_RET_OK && $totalDeals > 0) {
        $ret_page = $mt5->DealGetPage($trade_id, 0, 0, 0, $totalDeals, $deals);
        if ($ret_page == MTRetCode::MT_RET_OK) {
            foreach ($deals as $deal) {
                $volume += isset($deal->VolumeClosed) ? $deal->VolumeClosed * 0.0001 : 0;
            }
        }
    }

    // --- Floating P/L and Active Trades ---
    $floatingPL = 0;
    $activeTrades = 0;
    $positions = [];
    $totalPositions = 0;

    if ($mt5->PositionGetTotal($trade_id, $totalPositions) == MTRetCode::MT_RET_OK && $totalPositions > 0) {
        $ret_positions = $mt5->PositionGetPage($trade_id, 0, $totalPositions, $positions);
        if ($ret_positions == MTRetCode::MT_RET_OK) {
            foreach ($positions as $pos) {
                $floatingPL += floatval($pos->Profit ?? 0);
                $activeTrades++;
            }
        }
    }

    // --- Closed Trades Stats ---
    $closedTrades = 0;
    $profitableTrades = 0;
    $lossTrades = 0;
    $sumProfit = 0;

    $totalDeals = 0;
    $deals = [];
    $ret_total = $mt5->DealGetTotal($trade_id, 0, 0, $totalDeals);
    if ($ret_total == MTRetCode::MT_RET_OK && $totalDeals > 0) {
        $ret_page = $mt5->DealGetPage($trade_id, 0, 0, 0, $totalDeals, $deals);
        if ($ret_page == MTRetCode::MT_RET_OK && is_array($deals)) {
            foreach ($deals as $deal) {
                $order_id = $deal->Order ?? 0;
                $vol_closed = floatval($deal->VolumeClosed ?? 0) * 0.0001;

                if ($order_id != 0 && $vol_closed > 0) {
                    $profit = floatval($deal->Profit ?? 0);
                    $sumProfit += $profit;
                    $closedTrades++;
                    if ($profit > 0) $profitableTrades++;
                    elseif ($profit < 0) $lossTrades++;
                }
            }
        }
    }

    // --- Update liveaccounts volume ---
    $update_stmt = $conn->prepare("UPDATE liveaccounts SET volume=? WHERE trade_id=?");
    $update_stmt->bind_param("di", $volume, $trade_id);
    $update_stmt->execute();
    $update_stmt->close();

    // --- Deposits ---
    $stmtDep = $conn->prepare("SELECT SUM(deposit_amount) AS totalDeposit FROM wallet_deposit WHERE email = ? AND trade_id = ? AND Status=1");
    $stmtDep->bind_param("si", $email, $trade_id);
    $stmtDep->execute();
    $resDep = $stmtDep->get_result();
    $totalDeposit = floatval($resDep->fetch_assoc()['totalDeposit'] ?? 0);
    $stmtDep->close();

    // --- Withdrawals ---
    $stmtWith = $conn->prepare("SELECT SUM(withdraw_amount) AS totalWithdraw FROM wallet_withdraw WHERE email = ? AND trade_id = ? AND Status=1");
    $stmtWith->bind_param("si", $email, $trade_id);
    $stmtWith->execute();
    $resWith = $stmtWith->get_result();
    $totalWithdraw = floatval($resWith->fetch_assoc()['totalWithdraw'] ?? 0);
    $stmtWith->close();

    // --- Commission and Profit ---
    $isValidAccount = false;
    $account_type_name = $acc['account_type'] ?? '';
    $account_group = '';
    $commission_field_type = '';
    $rank_commission = 0;
    $profit_commission = 0;

    if (!empty($account_type_name)) {
        $stmt_type = $conn->prepare("SELECT * FROM account_types WHERE ac_name = ?");
        $stmt_type->bind_param("s", $account_type_name);
        $stmt_type->execute();
        $res_type = $stmt_type->get_result();

        if ($typeRow = $res_type->fetch_assoc()) {
            $account_type_name = $typeRow['ac_name'];
            $account_group = $typeRow['ac_group'];
            $isValidAccount = true;

            $commission_type = strtolower($account_type_name);
            $commission_field_type = "level_{$tierLevel}_{$commission_type}_commission";
            $rank_commission = isset($rankData[$commission_field_type])
                ? floatval($rankData[$commission_field_type])
                : floatval($rankData["level_{$tierLevel}_basic_commission"] ?? 1);

            $ib_comm_field = "ib_commission" . $tierLevel;
            $ib_percent = isset($typeRow[$ib_comm_field]) ? floatval($typeRow[$ib_comm_field]) : 0;

            if ($ib_percent > 0 && $sumProfit > 0) {
                $profit_commission = ($sumProfit * $ib_percent) / 100;
            }
        }
        $stmt_type->close();
    }

    $rebate = 0;
    if ($isValidAccount && $volume > 0) {
        $rebate = $volume * $rank_commission;
    }

    $name = '';
    $reg_date = $acc['Registered_Date'];
    $status = $acc['status'] ?? 'inactive';
    $stmt_user = $conn->prepare("SELECT first_name, last_name, reg_date, status FROM loginusers WHERE email = ?");
    $stmt_user->bind_param("s", $email);
    $stmt_user->execute();
    $res_user = $stmt_user->get_result();
    if ($userRow = $res_user->fetch_assoc()) {
        $name = trim(($userRow['first_name'] ?? '') . ' ' . ($userRow['last_name'] ?? ''));
        $reg_date = $userRow['reg_date'] ?? $reg_date;
        $status = isset($userRow['status']) ? ($userRow['status'] == 1 ? 'active' : 'inactive') : $status;
    }
    $stmt_user->close();

    $clients[] = [
        "trade_id" => $trade_id,
        "name" => $name,
        "email" => $email,
        "balance" => round($balance, 2),
        "account_type" => $account_type_name,
        "account_group" => $account_group,
        "deposit" => round($totalDeposit, 2),
        "withdraw" => round($totalWithdraw, 2),
        "volume" => round($volume, 2),
        "rebate" => round($rebate, 2),
        "rank_commission" => $rank_commission,
        "commission_field" => $commission_field_type,
        "floating_pl" => round($floatingPL, 2),
        "active_trades" => $activeTrades,
        "closed_trades" => $closedTrades,
        "profitable_trades" => $profitableTrades,
        "loss_trades" => $lossTrades,
        "sum_profit" => round($sumProfit, 2),
        "profit_commission" => round($profit_commission, 2),
        "reg_date" => $reg_date,
        "status" => $status
    ];

    $totals["total_balance"] += $balance;
    $totals["total_deposit"] += $totalDeposit;
    $totals["total_withdraw"] += $totalWithdraw;
    $totals["total_volume"] += $volume;
    $totals["total_rebate"] += $rebate;
    $totals["total_profit_commission"] += $profit_commission;
    $totals["total_clients"]++;
}

// ===== Calculate total profit commission for ALL levels (direct clients only) =====
$total_profit_commission_all_levels = 0;

for ($lvl = 1; $lvl <= 20; $lvl++) {
    $ibCol = "ib$lvl";
    $stmt_lvl = $conn->prepare("SELECT * FROM liveaccounts WHERE $ibCol = ?");
    $stmt_lvl->bind_param("s", $emailInput);
    $stmt_lvl->execute();
    $res_lvl = $stmt_lvl->get_result();
    $levelAccounts = $res_lvl->fetch_all(MYSQLI_ASSOC);
    $stmt_lvl->close();

    foreach ($levelAccounts as $acc) {
        $sumProfit = 0;
        $trade_id = intval($acc['trade_id']);
        $totalDeals = 0;
        $deals = [];
        $ret_total = $mt5->DealGetTotal($trade_id, 0, 0, $totalDeals);
        if ($ret_total == MTRetCode::MT_RET_OK && $totalDeals > 0) {
            $ret_page = $mt5->DealGetPage($trade_id, 0, 0, 0, $totalDeals, $deals);
            if ($ret_page == MTRetCode::MT_RET_OK && is_array($deals)) {
                foreach ($deals as $deal) {
                    $order_id = $deal->Order ?? 0;
                    $vol_closed = floatval($deal->VolumeClosed ?? 0) * 0.0001;
                    if ($order_id != 0 && $vol_closed > 0) {
                        $sumProfit += floatval($deal->Profit ?? 0);
                    }
                }
            }
        }

        // Get IB commission percent
        $stmt_type = $conn->prepare("SELECT * FROM account_types WHERE ac_name = ?");
        $stmt_type->bind_param("s", $acc['account_type']);
        $stmt_type->execute();
        $res_type = $stmt_type->get_result();
        if ($typeRow = $res_type->fetch_assoc()) {
            $ib_percent = 0;
            $field = "ib_commission$lvl";
            if (!empty($typeRow[$field])) {
                $ib_percent = floatval($typeRow[$field]);
            }
            if ($ib_percent > 0 && $sumProfit > 0) {
                $total_profit_commission_all_levels += ($sumProfit * $ib_percent) / 100;
            }
        }
        $stmt_type->close();
    }
}

// --- Insert or update total in ib_total_profit_commission ---
$stmt_profit = $conn->prepare("
    INSERT INTO ib_total_profit_commission
    (ib_email, total_profit_commission, available_profit_commission)
    VALUES (?, ?, ?)
    ON DUPLICATE KEY UPDATE
        total_profit_commission = VALUES(total_profit_commission),
        updated_at = CURRENT_TIMESTAMP
");
$available_profit_commission = $total_profit_commission_all_levels;
$stmt_profit->bind_param("sdd", $emailInput, $total_profit_commission_all_levels, $available_profit_commission);
$stmt_profit->execute();
$stmt_profit->close();

$mt5->Disconnect();

$response = [
    "status" => count($clients) > 0 ? "success" : "error",
    "data" => $clients,
    "totals" => $totals,
    "rank" => [
        "rank_number" => $rankNumber,
        "rank_name" => $rankName,
        "active_clients" => $totalActiveClients
    ]
];

if (count($clients) == 0) {
    $response["message"] = "No clients found for this email in this level.";
}

echo json_encode($response, JSON_PRETTY_PRINT);
?>
