<?php
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

include_once __DIR__ . "/config.php";
include_once __DIR__ . "/connection.php";
include_once __DIR__ . "/unified_config.php"; // For email system

$input = json_decode(file_get_contents("php://input"), true);
$action = $input['action'] ?? '';
$email = trim($input['email'] ?? '');

// All requests need email
if (empty($email)) {
    echo json_encode(["success" => false, "message" => "Email is required"]);
    exit;
}

// Helper: Generate 6-digit OTP
function generateOTP($length = 6) {
    return str_pad(rand(0, pow(10, $length)-1), $length, '0', STR_PAD_LEFT);
}

// Step 1: Initiate IB Withdrawal (Send OTP)
if ($action === 'initiate_ib_withdraw') {
    $amount = floatval($input['amount'] ?? 0);
    $walletaddress = trim($input['walletaddress'] ?? '');
    $network = trim($input['network'] ?? '');

    if ($amount <= 0) {
        echo json_encode(["success" => false, "message" => "Amount must be > 0"]);
        exit;
    }
    if (empty($walletaddress) || empty($network)) {
        echo json_encode(["success" => false, "message" => "Wallet address and network required"]);
        exit;
    }

    // Optional: Check if user has enough commission (you can add your logic here)
    // e.g., check total earned commission > amount

    $otp = generateOTP();
    $otp_expiry = time() + 600; // 10 minutes
    $random_ref = bin2hex(random_bytes(8));

    // Save to pending_ib_withdrawals
    $stmt = $conn->prepare("
        INSERT INTO pending_ib_withdrawals 
        (email, amount, walletaddress, network, reference, otp, otp_expiry)
        VALUES (?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->bind_param("sdssssi", $email, $amount, $walletaddress, $network, $random_ref, $otp, $otp_expiry);

    if ($stmt->execute()) {
        $pending_id = $conn->insert_id;

        // Send OTP Email
        $emailSent = false;
        if (EMAIL_AUTOMATION_ENABLED) {
            $stmtUser = $conn->prepare("SELECT first_name, last_name FROM loginusers WHERE email = ?");
            $stmtUser->bind_param("s", $email);
            $stmtUser->execute();
            $res = $stmtUser->get_result();
            if ($user = $res->fetch_assoc()) {
                $name = trim($user['first_name'] . ' ' . $user['last_name']);
                $mailSystem = getMailSystem();

                $mailResult = $mailSystem->sendWithdrawalOTP(
                    $email,
                    $name,
                    $otp,
                    $amount,
                    'USD',
                    "$network - $walletaddress",
                    '10 minutes'
                );

                $emailSent = $mailResult['success'] ?? (bool)$mailResult;
            }
            $stmtUser->close();
        }

        echo json_encode([
            "success" => true,
            "message" => "OTP sent to your email",
            "pending_id" => $pending_id,
            "email_sent" => $emailSent
        ]);
    } else {
        echo json_encode(["success" => false, "message" => "Failed to initiate withdrawal"]);
    }
    exit;
}

// Step 2: Verify OTP & Submit IB Withdrawal
if ($action === 'verify_ib_otp') {
    $pending_id = intval($input['pending_id'] ?? 0);
    $otp_input = trim($input['otp'] ?? '');

    if ($pending_id <= 0 || empty($otp_input)) {
        echo json_encode(["success" => false, "message" => "Invalid request"]);
        exit;
    }

    $stmt = $conn->prepare("SELECT * FROM pending_ib_withdrawals WHERE id = ? AND email = ?");
    $stmt->bind_param("is", $pending_id, $email);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($row = $result->fetch_assoc()) {
        if (time() > $row['otp_expiry']) {
            $conn->query("DELETE FROM pending_ib_withdrawals WHERE id = $pending_id");
            echo json_encode(["success" => false, "message" => "OTP expired"]);
            exit;
        }

        if ($otp_input !== $row['otp']) {
            echo json_encode(["success" => false, "message" => "Invalid OTP"]);
            exit;
        }

        // OTP correct → Create real withdrawal request
        $insert = $conn->prepare("
            INSERT INTO ib_withdraw_request 
            (email, total_commission, walletaddress, network, random, status)
            VALUES (?, ?, ?, ?, ?, 0)
        ");
        $insert->bind_param(
            "sdsss",
            $row['email'],
            $row['amount'],
            $row['walletaddress'],
            $row['network'],
            $row['reference']
        );

        if ($insert->execute()) {
            $request_id = $conn->insert_id;

            // Delete pending
            $conn->query("DELETE FROM pending_ib_withdrawals WHERE id = $pending_id");

            // Send confirmation email (optional)
            if (EMAIL_AUTOMATION_ENABLED) {
                $stmtUser = $conn->prepare("SELECT first_name, last_name FROM loginusers WHERE email = ?");
                $stmtUser->bind_param("s", $email);
                $stmtUser->execute();
                $res = $stmtUser->get_result();
                if ($user = $res->fetch_assoc()) {
                    $name = trim($user['first_name'] . ' ' . $user['last_name']);
                    $mailSystem = getMailSystem();
                    $mailSystem->sendWithdrawalRequestSubmitted(
                        $email, $name, $row['amount'], 'USD', $row['network'], 'IBW-'.$request_id, 'commission'
                    );
                }
            }

            echo json_encode([
                "success" => true,
                "message" => "IB withdrawal request submitted!",
                "request_id" => $request_id,
                "amount" => $row['amount'],
                "reference" => $row['reference']
            ]);
        } else {
            echo json_encode(["success" => false, "message" => "Failed to submit withdrawal"]);
        }
    } else {
        echo json_encode(["success" => false, "message" => "Invalid or expired session"]);
    }
    exit;
}

// Invalid action
echo json_encode(["success" => false, "message" => "Invalid action"]);
?>