<?php
/**
 * Final Email Integration Test
 * Tests all key email functionality after integration
 */

include_once __DIR__ . '/unified_config.php';

echo "=== ZForex Email Integration Test ===\n\n";

// Test 1: Check if mail system is properly loaded
echo "1. Testing Mail System Loading...\n";
if (class_exists('IntegratedMailSystem')) {
    echo "   ✅ IntegratedMailSystem class loaded successfully\n";
} else {
    echo "   ❌ IntegratedMailSystem class not found\n";
    exit(1);
}

// Test 2: Check if all required functions exist
echo "2. Testing Required Functions...\n";
if (function_exists('getMailSystem')) {
    echo "   ✅ getMailSystem() function available\n";
} else {
    echo "   ❌ getMailSystem() function not found\n";
}

if (function_exists('sendZForexMail')) {
    echo "   ✅ sendZForexMail() function available\n";
} else {
    echo "   ❌ sendZForexMail() function not found\n";
}

// Test 3: Initialize mail system
echo "3. Initializing Mail System...\n";
try {
    $mailSystem = getMailSystem();
    echo "   ✅ Mail system initialized successfully\n";
} catch (Exception $e) {
    echo "   ❌ Failed to initialize mail system: " . $e->getMessage() . "\n";
    exit(1);
}

// Test 4: Check configuration
echo "4. Checking Configuration...\n";
echo "   Mail Host: " . (defined('MAIL_HOST') ? MAIL_HOST : 'Not defined') . "\n";
echo "   Mail Port: " . (defined('MAIL_PORT') ? MAIL_PORT : 'Not defined') . "\n";
echo "   Mail Username: " . (defined('MAIL_USERNAME') ? MAIL_USERNAME : 'Not defined') . "\n";
echo "   Encryption: " . (defined('MAIL_ENCRYPTION') ? MAIL_ENCRYPTION : 'Not defined') . "\n";
echo "   Email Automation: " . (defined('EMAIL_AUTOMATION_ENABLED') ? (EMAIL_AUTOMATION_ENABLED ? 'Enabled' : 'Disabled') : 'Not defined') . "\n";

// Test 5: Test template availability
echo "5. Testing Template Availability...\n";
$reflection = new ReflectionClass($mailSystem);
$properties = $reflection->getProperty('templates');
$properties->setAccessible(true);
$templates = $properties->getValue($mailSystem);

$templateCount = count($templates);
echo "   Found $templateCount email templates\n";

if ($templateCount >= 22) {
    echo "   ✅ All 22+ templates loaded\n";
} else {
    echo "   ⚠️  Expected 22+ templates, found $templateCount\n";
}

// List key templates
$keyTemplates = [
    'register_verification',
    'login_alert',
    'open_live_account',
    'kyc_upload',
    'kyc_approved',
    'deposit_request_submitted',
    'deposit_approved',
    'deposit_failed',
    'withdrawal_request_submitted',
    'withdrawal_approved',
    'withdrawal_rejected'
];

echo "   Checking key templates:\n";
foreach ($keyTemplates as $template) {
    if (isset($templates[$template])) {
        echo "     ✅ $template\n";
    } else {
        echo "     ❌ $template\n";
    }
}

// Test 6: Test quick send methods
echo "6. Testing Quick Send Methods...\n";
$methodsToTest = [
    'sendRegisterVerification',
    'sendLoginAlert',
    'sendLiveAccountOpened',
    'sendKYCUpload',
    'sendKYCApproved',
    'sendDepositRequestSubmitted',
    'sendDepositApproved',
    'sendDepositFailed',
    'sendWithdrawalRequestSubmitted',
    'sendWithdrawalApproved',
    'sendWithdrawalRejected'
];

foreach ($methodsToTest as $method) {
    if (method_exists($mailSystem, $method)) {
        echo "   ✅ $method\n";
    } else {
        echo "   ❌ $method\n";
    }
}

echo "\n=== Test Complete ===\n";
echo "Email system is properly integrated with " . $templateCount . " templates.\n";
echo "All key functionality is available for use in API endpoints.\n";
?>